# license_prompt.py
import os

from PySide6.QtWidgets import (
    QDialog, QVBoxLayout, QHBoxLayout, QLabel, QLineEdit,
    QPushButton, QMessageBox, QWidget, QTextEdit, QComboBox,
    QTabWidget, QCheckBox, QSpacerItem, QProgressBar
)
from PySide6.QtCore import Qt, QTimer, QDate
from PySide6.QtGui import QPixmap, QIcon, QFont, QColor

# Use consistent relative imports for package structure
from .license_manager import get_trial_status, is_trial_active, load_trial_state

class LicensePromptDialog(QDialog):
    """Dialog to prompt user for license email and key with trial support"""
    
    def __init__(self, parent=None, trial_mode=False):
        super().__init__(parent)
        self.setWindowTitle("AIMMS License Activation")
        self.setModal(True)
        self.resize(600, 500)
        self.trial_mode = trial_mode
        
        # Create layout
        layout = QVBoxLayout()
        
        # Create tab widget for different activation methods
        self.tab_widget = QTabWidget()
        
        # Trial tab
        self.trial_tab = self.create_trial_tab()
        self.tab_widget.addTab(self.trial_tab, "Start Trial")
        
        # License file tab (Phase 4 - Asymmetric licensing)
        self.license_file_tab = self.create_license_file_tab()
        self.tab_widget.addTab(self.license_file_tab, "License File (.lic)")
        
        layout.addWidget(self.tab_widget)
        
        # Trial status panel (enhanced for Phase 3)
        self.trial_status_panel = self.create_trial_status_panel()
        layout.addWidget(self.trial_status_panel)
        
        # Status label
        self.status_label = QLabel("Please enter your license information or start a trial.")
        self.status_label.setAlignment(Qt.AlignCenter)
        self.status_label.setStyleSheet("font-size: 12px; color: palette(text);")
        layout.addWidget(self.status_label)
        
        # Buttons
        button_layout = QHBoxLayout()
        button_layout.addStretch()
        
        cancel_btn = QPushButton("Exit Licensing")
        cancel_btn.clicked.connect(self.reject)
        cancel_btn.setStyleSheet("background-color: #d32f2f; color: white;")
        
        help_btn = QPushButton("Help")
        help_btn.clicked.connect(self.show_help)
        
        activate_btn = QPushButton("Activate")
        activate_btn.setDefault(True)
        activate_btn.clicked.connect(self.activate_selected)
        activate_btn.setStyleSheet("background-color: #2e7d32; color: white;")
        
        button_layout.addWidget(cancel_btn)
        button_layout.addWidget(help_btn)
        button_layout.addWidget(activate_btn)
        
        layout.addLayout(button_layout)
        
        # Set up timer for trial status updates
        self.trial_timer = QTimer()
        self.trial_timer.timeout.connect(self.update_trial_status)
        self.trial_timer.timeout.connect(self.update_trial_tab_status)
        self.trial_timer.start(60000)  # Update every minute
        
        self.setLayout(layout)
        
        # Initialize trial status
        self.update_trial_status()
        self.update_trial_tab_status()
    
    
    def create_trial_tab(self):
        """Create the trial activation tab"""
        tab = QWidget()
        layout = QVBoxLayout()
        
        # Title
        title_label = QLabel("🚀 Start Free Trial")
        title_label.setAlignment(Qt.AlignCenter)
        title_label.setStyleSheet("font-weight: bold; font-size: 16px; margin: 10px;")
        layout.addWidget(title_label)
        
        # Information text
        info_text = QTextEdit()
        info_text.setReadOnly(True)
        info_text.setHtml("""
        <h3>30-Day Free Trial</h3>
        <p>Start your 30-day free trial to explore all features of AIMMS without any commitment.</p>
        
        <h4>Trial Benefits:</h4>
        <ul>
            <li>Full access to all features for 30 days</li>
            <li>No credit card required</li>
            <li>Easily upgrade to permanent license</li>
            <li>Perfect for evaluating the software</li>
        </ul>
        
        <h4>After Trial:</h4>
        <p>The application will require a valid license to continue running. You can purchase a permanent license at any time.</p>
        
        <h4><font color="red">Important:</font></h4>
        <p>The trial is bound to this machine and cannot be transferred.</p>
        """)
        layout.addWidget(info_text)
        
        # Email field for trial
        trial_email_layout = QHBoxLayout()
        trial_email_label = QLabel("Email Address:")
        trial_email_label.setFixedWidth(120)
        self.trial_email_edit = QLineEdit()
        self.trial_email_edit.setPlaceholderText("your-email@example.com")
        trial_email_layout.addWidget(trial_email_label)
        trial_email_layout.addWidget(self.trial_email_edit)
        layout.addLayout(trial_email_layout)
        
        layout.addSpacing(10)
        
        # Trial status
        self.trial_status_label = QLabel("Trial status: Checking...")
        self.trial_status_label.setAlignment(Qt.AlignCenter)
        self.trial_status_label.setStyleSheet("font-size: 12px; color: #666;")
        layout.addWidget(self.trial_status_label)
        
        layout.addStretch()
        
        tab.setLayout(layout)
        return tab

    def create_license_file_tab(self):
        """Create the license file activation tab (Phase 4)"""
        tab = QWidget()
        layout = QVBoxLayout()
        
        # Title
        title_label = QLabel("📄 License File Activation")
        title_label.setAlignment(Qt.AlignCenter)
        title_label.setStyleSheet("font-weight: bold; font-size: 16px; margin: 10px;")
        layout.addWidget(title_label)
        
        # Information text
        info_text = QTextEdit()
        info_text.setReadOnly(True)
        info_text.setHtml("""
        <h3>Activate with License File</h3>
        <p>If you have received a .lic license file, you can activate AIMMS by selecting the file.</p>
        
        <h4>Benefits of License File Activation:</h4>
        <ul>
            <li>Secure asymmetric cryptography</li>
            <li>No manual key entry required</li>
            <li>Automatic validation</li>
            <li>Supports both permanent and trial licenses</li>
        </ul>
        
        <h4>How to use:</h4>
        <p>1. Click the "Browse..." button below</p>
        <p>2. Select your .lic license file</p>
        <p>3. Click "Activate" to validate and activate your license</p>
        
        <h4>Note:</h4>
        <p>License files use RSA-PSS-SHA256 digital signatures for maximum security.</p>
        """)
        layout.addWidget(info_text)
        
        # License file selection
        file_layout = QHBoxLayout()
        file_label = QLabel("License File:")
        file_label.setFixedWidth(120)
        self.license_file_edit = QLineEdit()
        self.license_file_edit.setPlaceholderText("Select a .lic file")
        self.license_file_edit.setReadOnly(True)
        
        browse_btn = QPushButton("Browse...")
        browse_btn.clicked.connect(self.browse_license_file)
        
        file_layout.addWidget(file_label)
        file_layout.addWidget(self.license_file_edit)
        file_layout.addWidget(browse_btn)
        layout.addLayout(file_layout)
        
        layout.addSpacing(15)
        
        # Status label for file validation
        self.file_status_label = QLabel("")
        self.file_status_label.setAlignment(Qt.AlignCenter)
        self.file_status_label.setStyleSheet("font-size: 12px; color: #666;")
        layout.addWidget(self.file_status_label)
        
        layout.addStretch()
        
        tab.setLayout(layout)
        return tab
    
    def create_trial_status_panel(self):
        """Create enhanced license status panel with visual indicators and progress"""
        panel = QWidget()
        layout = QVBoxLayout()
        
        # Title
        title_label = QLabel("📊 License Status")
        title_label.setAlignment(Qt.AlignCenter)
        title_label.setStyleSheet("font-weight: bold; font-size: 14px; margin: 5px;")
        layout.addWidget(title_label)
        
        # Trial status display
        self.trial_status_display = QLabel("Checking trial status...")
        self.trial_status_display.setAlignment(Qt.AlignCenter)
        self.trial_status_display.setStyleSheet("font-size: 12px; margin: 5px;")
        layout.addWidget(self.trial_status_display)
        
        # Progress bar for trial days
        self.trial_progress_bar = QProgressBar()
        self.trial_progress_bar.setRange(0, 30)
        self.trial_progress_bar.setValue(0)
        self.trial_progress_bar.setTextVisible(True)
        self.trial_progress_bar.setFormat("%d / 30 days")
        layout.addWidget(self.trial_progress_bar)
        
        # Days remaining label
        self.days_remaining_label = QLabel("Days remaining: --")
        self.days_remaining_label.setAlignment(Qt.AlignCenter)
        self.days_remaining_label.setStyleSheet("font-size: 11px; color: #666; margin: 2px;")
        layout.addWidget(self.days_remaining_label)
        
        # Warning label for expiring trial
        self.trial_warning_label = QLabel("")
        self.trial_warning_label.setAlignment(Qt.AlignCenter)
        self.trial_warning_label.setWordWrap(True)
        layout.addWidget(self.trial_warning_label)
        
        # Upgrade prompt button (initially hidden)
        self.upgrade_prompt_btn = QPushButton("🚀 Upgrade to Permanent License")
        self.upgrade_prompt_btn.clicked.connect(self.show_license_activation)
        self.upgrade_prompt_btn.setStyleSheet("""
            QPushButton {
                background-color: #ff9800;
                color: white;
                border: none;
                border-radius: 4px;
                padding: 5px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #f57c00;
            }
        """)
        self.upgrade_prompt_btn.hide()
        layout.addWidget(self.upgrade_prompt_btn)
        
        layout.addStretch()
        panel.setLayout(layout)
        return panel
    
    def show_license_activation(self):
        """Switch to license activation tab when upgrade button is clicked"""
        self.tab_widget.setCurrentIndex(0)  # Switch to License tab
        self.status_label.setText("🚀 Click 'Activate' to upgrade your license and continue using all features.")
        self.status_label.setStyleSheet("font-size: 12px; color: #ff9800; font-weight: bold;")
        self.upgrade_prompt_btn.show()
    
    def update_trial_status(self):
        """Update the enhanced trial status display with visual indicators"""
        try:
            # First check if we have a valid license (use package-relative import)
            from .license_manager import load_license, validate_license_production_with_guardrails
            license_data = load_license()
            
            if license_data:
                email = license_data.get("email")
                key = license_data.get("key")
                if email and key:
                    # Try the new asymmetric validation system first
                    try:
                        is_valid, message, integrity_failed = validate_license_production_with_guardrails("1.0")
                        if is_valid:
                            # Show fully licensed status
                            self.trial_status_display.setText(f"✅ Fully Licensed to: {email}")
                            self.trial_status_display.setStyleSheet("font-size: 12px; color: #2e7d32; font-weight: bold;")
                            self.trial_progress_bar.setValue(30)
                            self.days_remaining_label.setText("Unlimited access")
                            self.trial_warning_label.setText("Thank you for supporting AIMMS!")
                            self.trial_warning_label.setStyleSheet("font-size: 11px; color: #2e7d32;")
                            self.upgrade_prompt_btn.hide()
                            return
                    except Exception:
                        # Fall back to old validation system - removed as part of legacy cleanup
                        pass
            
            # If no valid license, check trial status
            trial_active, message, days_remaining = get_trial_status()
             
            # Update main trial status display
            if trial_active:
                self.trial_status_display.setText(f"✅ {message}")
                self.trial_status_display.setStyleSheet("font-size: 12px; color: #2e7d32;")
                 
                # Update progress bar
                if days_remaining >= 0:
                    self.trial_progress_bar.setValue(days_remaining)
                    self.days_remaining_label.setText(f"Days remaining: {days_remaining}")
                     
                    # Update progress bar color based on remaining days
                    if days_remaining <= 3:
                        self.trial_progress_bar.setStyleSheet("""
                            QProgressBar {
                                border: 1px solid #ccc;
                                border-radius: 3px;
                                text-align: center;
                                background-color: #ffebee;
                            }
                            QProgressBar::chunk {
                                background-color: #f44336;
                            }
                        """)
                        self.trial_warning_label.setText("⚠️ Trial expiring soon! Upgrade to continue using all features.")
                        self.trial_warning_label.setStyleSheet("font-size: 11px; color: #d32f2f; font-weight: bold;")
                        self.upgrade_prompt_btn.show()
                    elif days_remaining <= 7:
                        self.trial_progress_bar.setStyleSheet("""
                            QProgressBar {
                                border: 1px solid #ccc;
                                border-radius: 3px;
                                text-align: center;
                                background-color: #fff3e0;
                            }
                            QProgressBar::chunk {
                                background-color: #ff9800;
                            }
                        """)
                        self.trial_warning_label.setText("⚠️ Trial expiring soon! Consider upgrading to continue using all features.")
                        self.trial_warning_label.setStyleSheet("font-size: 11px; color: #ff9800;")
                        self.upgrade_prompt_btn.show()
                    else:
                        self.trial_progress_bar.setStyleSheet("""
                            QProgressBar {
                                border: 1px solid #ccc;
                                border-radius: 3px;
                                text-align: center;
                                background-color: #e8f5e8;
                            }
                            QProgressBar::chunk {
                                background-color: #4caf50;
                            }
                        """)
                        self.trial_warning_label.setText("")
                        self.upgrade_prompt_btn.hide()
                else:
                    self.trial_progress_bar.setValue(0)
                    self.days_remaining_label.setText("Days remaining: 0")
                    self.trial_warning_label.setText("❌ Trial expired. Please upgrade to continue using AIMMS.")
                    self.trial_warning_label.setStyleSheet("font-size: 11px; color: #d32f2f; font-weight: bold;")
                    self.upgrade_prompt_btn.show()
                 
            else:
                self.trial_status_display.setText(f"❌ {message}")
                self.trial_status_display.setStyleSheet("font-size: 12px; color: #d32f2f;")
                self.trial_progress_bar.setValue(0)
                self.days_remaining_label.setText("Days remaining: --")
                self.trial_warning_label.setText("")
                self.upgrade_prompt_btn.hide()
               
        except PermissionError as pe:
            # Specific handling for permission errors when accessing trial files
            msg = f"❌ Permission denied checking trial status: {pe}"
            print(msg)
            self.trial_status_display.setText(msg)
            self.trial_status_display.setStyleSheet("font-size: 12px; color: #d32f2f;")
            self.trial_progress_bar.setValue(0)
            self.days_remaining_label.setText("Days remaining: --")
            self.trial_warning_label.setText("")
            self.upgrade_prompt_btn.hide()
        except Exception as e:
            # Show the exception message to help debugging permissions/IO issues
            import traceback
            tb = traceback.format_exc()
            print(f"Error in update_trial_status: {e}\n{tb}")
            self.trial_status_display.setText(f"❌ Unable to check trial status: {str(e)}")
            self.trial_status_display.setStyleSheet("font-size: 12px; color: #d32f2f;")
            self.trial_progress_bar.setValue(0)
            self.days_remaining_label.setText("Days remaining: --")
            self.trial_warning_label.setText("")
            self.upgrade_prompt_btn.hide()

    def update_trial_tab_status(self):
        """Update the trial status label in the trial tab"""
        try:
            # Check trial status
            try:
                trial_active, message, days_remaining = get_trial_status()
            except PermissionError as pe:
                self.trial_status_label.setText(f"Trial status: Permission denied: {pe}")
                self.trial_status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
                return

            if trial_active:
                self.trial_status_label.setText(f"Trial status: Active ({days_remaining} days remaining)")
                self.trial_status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
            else:
                self.trial_status_label.setText("Trial status: Not started")
                self.trial_status_label.setStyleSheet("font-size: 12px; color: #666;")

        except Exception as e:
            import traceback
            print(f"Error in update_trial_tab_status: {e}\n{traceback.format_exc()}")
            self.trial_status_label.setText("Trial status: Error checking status")
            self.trial_status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
    
    def browse_license_file(self):
        """Browse for and select a license file"""
        file_path = show_license_file_prompt(self)
        if file_path:
            self.license_file_edit.setText(file_path)
            self.file_status_label.setText("✅ License file selected")
            self.file_status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
            
            # Validate the license file
            try:
                # Import required modules
                import sys
                import importlib.util
                import platform
                
                # Define function to get license resource path with cross-platform compatibility
                def get_license_resource_path(filename):
                    """Get path to licensing resource with cross-platform compatibility"""
                    try:
                        # Check if we're running in a compiled executable
                        if getattr(sys, 'frozen', False):
                            # Nuitka build - try multiple possible locations
                            possible_paths = [
                                os.path.join(os.path.dirname(sys.executable), filename),
                                os.path.join(os.path.dirname(os.path.abspath(__file__)), filename),
                                filename
                            ]
                            
                            for path in possible_paths:
                                if os.path.exists(path):
                                    return path
                        
                        # Fallback to normal path resolution
                        return os.path.join(os.path.dirname(os.path.abspath(__file__)), filename)
                        
                    except Exception:
                        # Final fallback
                        return filename
                
                # Load license_core_new module directly from file
                expected_path = get_license_resource_path("license_core_new.py")
                spec = importlib.util.spec_from_file_location("license_core_new", expected_path)
                license_core_new_module = importlib.util.module_from_spec(spec)
                sys.modules["license_core_new"] = license_core_new_module
                spec.loader.exec_module(license_core_new_module)
                
                # Get the validate_license function
                validate_license = getattr(license_core_new_module, 'validate_license')
                
                # Validate the license file
                is_valid, message, license_data = validate_license(file_path)
                
                if is_valid:
                    self.file_status_label.setText(f"✅ Valid license: {message}")
                    self.file_status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
                    
                    # Clear the trial status message that was set in activate_license_from_trial()
                    self.status_label.setText("✅ License file validated and ready for activation.")
                    self.status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
                    
                    # Store license info for activation
                    email = license_data.get("email", "")
                    if email:
                        self.email = email
                        self.license_file_path = file_path
                        self.activation_type = "license_file"
                else:
                    self.file_status_label.setText(f"❌ Invalid license: {message}")
                    self.file_status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
                    
            except Exception as e:
                self.file_status_label.setText(f"❌ Error validating license: {str(e)}")
                self.file_status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
        else:
            self.file_status_label.setText("")

    def activate_selected(self):
        """Activate based on selected tab"""
        current_tab = self.tab_widget.currentIndex()
        
        if current_tab == 0:  # Trial tab
            self.start_trial()
        elif current_tab == 1:  # License file tab
            self.activate_license_file()
    
    
    def start_trial(self):
        """Start a new trial"""
        email = self.trial_email_edit.text().strip()
        
        if not email:
            self.status_label.setText("❌ Please enter your email address for the trial.")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
            return
        
        # Basic email format check
        if "@" not in email or "." not in email:
            self.status_label.setText("❌ Please enter a valid email address.")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
            return
        
        # Store the trial email
        self.email = email
        self.activation_type = "trial"
        
        self.status_label.setText("✅ Starting trial...")
        self.status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
        
        self.accept()
    
    def activate_license_file(self):
        """Activate license using selected .lic file"""
        file_path = self.license_file_edit.text().strip()
        
        if not file_path:
            self.status_label.setText("❌ Please select a license file.")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
            return
        
        if not file_path.endswith('.lic'):
            self.status_label.setText("❌ Please select a valid .lic license file.")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
            return
        
        # Validate the license file one more time
        try:
            from license_core_new import validate_license
            is_valid, message, license_data = validate_license(file_path)
            if is_valid:
                # Store license info for activation
                email = license_data.get("email", "")
                if email:
                    self.email = email
                    self.license_file_path = file_path
                    self.activation_type = "license_file"
                    
                    self.status_label.setText("✅ License file validated. Activating...")
                    self.status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
                    
                    self.accept()
                else:
                    self.status_label.setText("❌ Invalid license file: No email found")
                    self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
            else:
                self.status_label.setText(f"❌ Invalid license file: {message}")
                self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
        except Exception as e:
            self.status_label.setText(f"❌ Error validating license file: {str(e)}")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")

    def get_license_info(self):
        """Return the activation information"""
        if hasattr(self, 'email') and hasattr(self, 'activation_type'):
            if self.activation_type == "trial":
                return self.email
            elif self.activation_type == "license_file" and hasattr(self, 'license_file_path'):
                return self.license_file_path  # Return file path for license file activation
        return None
    
    def show_help(self):
        """Show help information"""
        help_text = """
        <h3>License Activation Help</h3>
        
        <h4>AIMMS Licensing Options</h4>
        <p>AIMMS offers two licensing methods:</p>
        
        <h4>1. License File Activation</h4>
        <p>If you have purchased a license, use the "License File (.lic)" tab to activate AIMMS with your .lic file.</p>
        <ul>
            <li>Secure asymmetric cryptography</li>
            <li>No manual key entry required</li>
            <li>Supports both permanent and trial licenses</li>
        </ul>
        
        <h4>2. Free Trial Activation</h4>
        <p>Start a 30-day free trial to evaluate AIMMS before purchasing. No license file required.</p>
        <ul>
            <li>Full access to all features</li>
            <li>No credit card required</li>
            <li>Bound to this machine</li>
            <li>Easily upgrade to permanent license</li>
        </ul>
        
        <h4>Getting a License</h4>
        <p>To obtain a permanent license:</p>
        <ul>
            <li>Contact the developer for licensing information</li>
            <li>Visit our website to purchase a license</li>
            <li>You will receive a .lic file for activation</li>
        </ul>
        
        <h4>Trial Information</h4>
        <p>The 30-day trial provides:</p>
        <ul>
            <li>Complete access to all features</li>
            <li>Free evaluation period</li>
            <li>Perfect for testing AIMMS workflow</li>
            <li>Can be upgraded at any time</li>
        </ul>
        
        <h4>Need Support?</h4>
        <p>If you need assistance with licensing, please contact support for help with trial activation or license file issues.</p>
        """
        
        QMessageBox.information(self, "License Help", help_text)

    def show_upgrade_progress(self, in_progress=True):
        """Show upgrade progress indicator (Phase 3)"""
        if in_progress:
            self.status_label.setText("🔄 Processing license activation...")
            self.status_label.setStyleSheet("font-size: 12px; color: #ff9800;")
            # Disable buttons during processing
            self.upgrade_prompt_btn.setEnabled(False)
        else:
            self.status_label.setText("✅ Ready to activate license")
            self.status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
            # Re-enable buttons
            self.upgrade_prompt_btn.setEnabled(True)

    def activate_license_from_trial(self):
        """Activate license from trial with enhanced flow (Phase 3)"""
        try:
            # Show progress
            self.show_upgrade_progress(True)
            
            # Get current trial email
            trial_data = load_trial_state()
            if not trial_data:
                self.status_label.setText("❌ No active trial found")
                self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
                return
            
            trial_email = trial_data.get('email')
            if not trial_email:
                self.status_label.setText("❌ Trial email not found")
                self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")
                return
            
            # Pre-fill email field
            self.trial_email_edit.setText(trial_email)
            self.trial_email_edit.setEnabled(False)  # Lock email field
            
            # Switch to license file tab (tab index 1)
            self.tab_widget.setCurrentIndex(1)
            
            # Update status
            self.status_label.setText("") # cleared to stop other messages popping up in license dialogue area.
            self.status_label.setStyleSheet("font-size: 12px; color: #2e7d32;")
            
            # Focus on the browse button for license file selection
            # Find the browse button in the license file tab
            for widget in self.license_file_tab.findChildren(QPushButton):
                if widget.text() == "Browse...":
                    widget.setFocus()
                    break
            
        except Exception as e:
            self.status_label.setText(f"❌ Error: {str(e)}")
            self.status_label.setStyleSheet("font-size: 12px; color: #d32f2f;")

def show_license_prompt(parent=None, trial_mode=False):
    """
    Show the license prompt dialog and return (email, key) or email for trial
    Enhanced with trial upgrade support (Phase 3) and asymmetric licensing (Phase 4)
    """
    try:
        # Create dialog with or without parent
        dialog = LicensePromptDialog(parent, trial_mode)
        
        # Set dialog flags to work without parent
        if parent is None:
            dialog.setWindowFlags(dialog.windowFlags() | Qt.Dialog | Qt.WindowStaysOnTopHint)
            dialog.setModal(True)
        
        # If trial is active, offer to upgrade
        if is_trial_active():
            dialog.activate_license_from_trial()
            
        result = dialog.exec()
        
        if result == QDialog.Accepted:
            return dialog.get_license_info()
        else:
            return None
            
    except Exception as e:
        print(f"Error showing license prompt: {e}")
        # Fallback: try to handle license file directly
        if parent is None:
            return handle_license_file_fallback()
        return None

def handle_license_file_fallback():
    """Fallback function to handle license file when dialog fails"""
    try:
        from PySide6.QtWidgets import QFileDialog, QMessageBox
        from pathlib import Path
        
        # Show file dialog to select license file
        file_path, _ = QFileDialog.getOpenFileName(
            None,
            "Select License File",
            "",
            "License Files (*.lic);;All Files (*)"
        )
        
        if file_path and file_path.endswith('.lic'):
            # Try to activate license file directly
            try:
                from licensing.license_manager import activate_license_file
                success = activate_license_file(file_path)
                if success:
                    # Extract license info
                    from licensing.license_core_new import LicenseCore
                    core = LicenseCore()
                    license_info = core.extract_license_info(file_path)
                    return license_info['email'], license_info['key']
                else:
                    QMessageBox.critical(
                        None,
                        "License Activation Failed",
                        "Failed to activate the selected license file."
                    )
            except Exception as e:
                QMessageBox.critical(
                    None,
                    "License Error",
                    f"Error processing license file: {e}"
                )
        return None
        
    except Exception as e:
        print(f"Error in license file fallback: {e}")
        return None


def show_license_file_prompt(parent=None):
    """
    Show dialog to prompt user for license file (.lic format)
    Returns path to license file if selected, None otherwise
    """
    from PySide6.QtWidgets import QFileDialog, QMessageBox
    
    # Show file dialog to select .lic file
    file_dialog = QFileDialog(parent)
    file_dialog.setWindowTitle("Select License File")
    file_dialog.setNameFilter("License Files (*.lic)")
    file_dialog.setFileMode(QFileDialog.ExistingFile)
    
    if file_dialog.exec() == QFileDialog.Accepted:
        selected_files = file_dialog.selectedFiles()
        if selected_files:
            return selected_files[0]
    
    return None