# license_manager.py
import json
import os
import sys
import datetime
import hashlib
import shutil
from typing import Optional, Tuple

# Use consistent relative imports for package structure
from .license_core import validate_trial_license, save_trial_state, load_trial_state, get_stable_machine_fingerprint
from .secure_revocation import SecureRevocationManager
from .trial_integrity_manager import TrialIntegrityManager

LICENSE_FILE = os.path.join(
    os.path.expanduser("~"),
    ".aimms_license.json"
)

def load_license():
    """Load license from file"""
    if not os.path.exists(LICENSE_FILE):
        return None
    try:
        with open(LICENSE_FILE, "r") as f:
            return json.load(f)
    except Exception as e:
        print(f"Error loading license file: {e}")
        return None

def save_license(email, key):
    """Save license to file"""
    try:
        with open(LICENSE_FILE, "w") as f:
            json.dump({"email": email, "key": key}, f)
        return True
    except Exception as e:
        print(f"Error saving license file: {e}")
        return False

def validate_license_with_trial(email: str, key: str, app_version: str) -> tuple[bool, str, bool, bool, bool]:
    """
    Validate license with trial support

    Returns:
        tuple[bool, str, bool, bool, bool]: (is_valid, message, is_permanent, is_trial_active, is_expired)
    """
    email = email.strip().lower()
    key = key.replace("-", "").strip() if key else ""

    # Check if permanent license first using new asymmetric validation
    try:
        from license_core_new import validate_license_key
        is_valid, message, license_data = validate_license_key(email, key)
        if is_valid:
            license_type = license_data.get("type", "")
            if license_type == "permanent":
                # If we have a valid license, delete any existing trial state to prevent conflicts
                delete_trial_state()
                return True, message, True, False, False
    except ImportError:
        # Fallback to old method if new validation system is not available
        pass

    # Check trial license using old method (keep existing trial system intact)
    trial_valid, trial_message, is_permanent, is_expired = validate_trial_license(email, key)

    if trial_valid:
        # If trial is active and not expired
        if not is_expired:
            return True, trial_message, False, True, False
        # If trial is expired
        else:
            return False, trial_message, False, False, True

    # If no valid license or trial found
    return False, "No valid license or trial found", False, False, False

def should_enable_paid_features(email: str, key: str, app_version: str) -> bool:
    """Determine if paid features should be enabled"""
    is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)

    if not is_valid:
        return False

    # Enable features for both permanent licenses and active trials
    if is_permanent or is_trial_active:
        return True

    return False

def get_trial_feature_restrictions():
    """Get list of features that are restricted during trial or after expiration"""
    # Return empty list - no restrictions during trial period
    return []

def get_trial_mode_restrictions():
    """Get comprehensive trial mode restrictions and limitations"""
    # Return empty restrictions - full access during trial
    return {
        "feature_restrictions": [],
        "usage_limits": {},
        "access_restrictions": [],
        "watermarking": False,
        "support_level": "full"
    }

def is_feature_allowed(feature_name: str, email: str = None, key: str = None, app_version: str = "1.0") -> bool:
    """
    Check if a specific feature is allowed based on license status

    Args:
        feature_name: Name of the feature to check
        email: User email (optional, will check from license file if not provided)
        key: License key (optional, will check from license file if not provided)
        app_version: Application version for validation

    Returns:
        bool: True if feature is allowed, False if restricted
    """
    # Get trial/limited features (now empty list)
    trial_limited_features = get_trial_feature_restrictions()

    # Since trial_limited_features is now empty, all features are allowed
    # But we still need to check license status
    if email and key:
        # Use provided credentials
        is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
    else:
        # Check from license file
        from .license_core import load_license
        license_data = load_license()
        if license_data:
            email = license_data.get("email")
            key = license_data.get("key")
            is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
        else:
            # No license found - check for active trial
            trial_active, trial_message, days_remaining = get_trial_status()
            if trial_active:
                return True  # Allow features for active trial
            else:
                return False  # Block features if no license and no active trial

    # Allow features for permanent licenses and active trials only
    if is_valid and (is_permanent or is_trial_active):
        return True
    
    # Block features for expired trials or invalid licenses
    return False

def check_trial_mode_compliance(feature_name: str, usage_count: int = None, max_limit: int = None) -> tuple[bool, str]:
    """
    Check if a feature is compliant with trial mode restrictions

    Args:
        feature_name: Name of the feature to check
        usage_count: Current usage count (for quota-based restrictions)
        max_limit: Maximum allowed limit (for quota-based restrictions)

    Returns:
        tuple[bool, str]: (is_compliant, message)
    """
    try:
        from .license_core import load_license, load_trial_state

        # Check license status
        license_data = load_license()
        has_license = license_data is not None

        # Check trial state
        trial_data = load_trial_state()
        has_trial = trial_data is not None

        # Check trial status
        trial_active, trial_message, days_remaining = get_trial_status()

        # If permanent license, allow everything
        if has_license:
            from .license_core_new import validate_license_key
            email = license_data.get("email")
            key = license_data.get("key")
            if email and key:
                is_valid, message, license_data_new = validate_license_key(email, key)
                if is_valid:
                    return True, "Permanent license - all features available"

        # Block features if no valid license or trial
        return False, "Feature blocked - no valid license or trial"

    except Exception:
        return False, "Feature blocked - license validation error"

def should_show_upgrade_prompt(feature_name: str) -> bool:
    """
    Determine if an upgrade prompt should be shown for a restricted feature

    Args:
        feature_name: Name of the feature being accessed

    Returns:
        bool: True if upgrade prompt should be shown
    """
    # Since we have no restrictions during trial, never show upgrade prompts
    return False

def create_upgrade_prompt_message(feature_name: str) -> str:
    """
    Create an upgrade prompt message for a restricted feature

    Args:
        feature_name: Name of the feature being restricted

    Returns:
        str: Upgrade prompt message
    """
    # Return empty message since we don't restrict any features
    return ""

def get_license_status_summary() -> dict:
    """
    Get comprehensive license status summary for UI display

    Returns:
        dict: Contains license status information
    """
    try:
        from .license_core import load_license, load_trial_state

        # Check license file
        license_data = load_license()
        has_license = license_data is not None

        # Check trial state
        trial_data = load_trial_state()
        has_trial = trial_data is not None

        # Check trial status
        trial_active, trial_message, days_remaining = get_trial_status()

        # Check permanent license
        license_valid = False
        license_message = ""
        if has_license:
            from .license_core_new import validate_license_key
            email = license_data.get("email")
            key = license_data.get("key")
            if email and key:
                is_valid, message, license_data_new = validate_license_key(email, key)
                license_valid = is_valid
                license_message = "Permanent license" if license_valid else "Invalid license"

        # Determine overall status - PRIORITIZE LICENSE OVER TRIAL
        if license_valid:
            status = "licensed"
            status_message = license_message
            days_remaining = None  # Not applicable for permanent license
        elif trial_active:
            status = "trial_active"
            status_message = trial_message
        elif has_trial:
            status = "trial_expired"
            status_message = "Trial expired"
            days_remaining = 0
        else:
            status = "unlicensed"
            status_message = "No license or trial found"
            days_remaining = None

        return {
            "status": status,
            "status_message": status_message,
            "has_license": has_license,
            "has_trial": has_trial,
            "trial_active": trial_active,
            "days_remaining": days_remaining,
            "license_valid": license_valid,
            "restricted_features": []  # No restricted features
        }

    except Exception as e:
        return {
            "status": "error",
            "status_message": f"Error checking license: {str(e)}",
            "has_license": False,
            "has_trial": False,
            "trial_active": False,
            "days_remaining": None,
            "license_valid": False,
            "restricted_features": []
        }

def is_trial_active() -> bool:
    """Check if trial is currently active"""
    trial_data = load_trial_state()
    if not trial_data:
        return False

    current_date = datetime.datetime.now()
    trial_end_date = datetime.datetime.fromisoformat(trial_data['end_date'])

    return current_date <= trial_end_date

def get_trial_status() -> tuple[bool, str, int]:
    """
    Get trial status with days remaining using secure trial state manager

    Returns:
        tuple[bool, str, int]: (is_active, message, days_remaining)
    """
    try:
        from license_core import _trial_state_manager
        result = _trial_state_manager.get_trial_status()
        return result
    except ImportError:
        # Fallback to old method
        trial_data = load_trial_state()
        if not trial_data:
            return False, "No trial found", 0

        current_date = datetime.datetime.now()
        trial_end_date = datetime.datetime.fromisoformat(trial_data['end_date'])

        if current_date > trial_end_date:
            return False, "Trial expired", 0

        days_remaining = (trial_end_date - current_date).days
        status_msg = f"Trial active - {days_remaining} days remaining"
        return True, status_msg, days_remaining

def start_trial(email: str) -> bool:
    """Start a new trial period"""
    # Check if trial already exists
    trial_data = load_trial_state()

    # Compute current machine fingerprint hash for comparison
    try:
        current_fp = get_stable_machine_fingerprint()
        current_hash = hashlib.sha256(current_fp.encode()).hexdigest()[:16]
    except Exception:
        current_hash = None

    if trial_data:
        # Try to read stored fingerprint hash (new format)
        stored_hash = trial_data.get('machine_fingerprint_hash') or trial_data.get('machine_fingerprint')

        # If stored_hash exists and does not match current machine, prevent reuse on this machine
        if stored_hash and current_hash and stored_hash != current_hash:
            print("❌ Trial exists but is bound to a different machine; cannot start new trial on this machine")
            return False

        # If trial is still active, don't restart
        try:
            current_date = datetime.datetime.now()
            trial_end_date = datetime.datetime.fromisoformat(trial_data.get('end_date'))
            if current_date <= trial_end_date:
                return False
        except Exception:
            # If trial data malformed, do not allow automatic restart - require manual intervention
            print("Warning: Existing trial data malformed; cannot automatically start a new trial")
            return False

        # If trial expired and machine matches, DO NOT allow restarting trial per policy
        print("❌ Trial previously used on this machine and expired; cannot start a new trial")
        return False

    # Save new trial state (no existing trial found)
    result = save_trial_state(email)

    # Verify trial was saved
    new_trial_data = load_trial_state()

    return result

def delete_trial_state(confirm: bool = False):
    """Delete trial state from all secure locations"""
    # Keep backward compatible simple delete
    try:
        from logging_manager import log_boot_message, log_boot_error
    except Exception:
        log_boot_message = log_boot_error = None

    try:
        # If confirmation requested and we are running in a Qt context, ask the user
        if confirm:
            try:
                from PySide6.QtWidgets import QMessageBox
                resp = QMessageBox.question(None, "Confirm Delete Trial", "Are you sure you want to delete the local trial state? This cannot be undone.", QMessageBox.Yes | QMessageBox.No)
                if resp != QMessageBox.Yes:
                    if log_boot_message:
                        log_boot_message("User cancelled trial state deletion")
                    return False
            except Exception:
                # If UI not available, fall through to non-interactive path
                pass
        from license_core import delete_trial_state as core_delete_trial_state, load_trial_state
    except ImportError:
        print("Warning: Could not import delete_trial_state from license_core")
        return False

    # Integrity check - ensure trial state belongs to this machine before deleting
    try:
        trial_data = load_trial_state()
        if trial_data:
            stored_hash = trial_data.get('machine_fingerprint_hash') or trial_data.get('machine_fingerprint')
            try:
                current_fp = get_stable_machine_fingerprint()
                current_hash = hashlib.sha256(current_fp.encode()).hexdigest()[:16]
            except Exception:
                current_hash = None

            if stored_hash and current_hash and stored_hash != current_hash:
                # Log and refuse deletion - protects against deleting other-machine trials
                msg = "Refusing to delete trial state: trial bound to different machine"
                if log_boot_error:
                    log_boot_error(msg)
                else:
                    print(msg)
                return False

        # Proceed to delete
        if log_boot_message:
            log_boot_message("Attempting to delete trial state (requested)")
        result = core_delete_trial_state()
        if result:
            if log_boot_message:
                log_boot_message("Trial state deleted successfully")
            return True
        else:
            if log_boot_error:
                log_boot_error("Trial state deletion reported failure")
            return False
    except Exception as e:
        if log_boot_error:
            log_boot_error(f"Error deleting trial state: {e}")
        else:
            print(f"Error deleting trial state: {e}")
        return False

def validate_license_production_with_guardrails(app_version: str = "1.0") -> tuple[bool, str, bool]:
    """Validate the license file with all guardrails - now supports both old and new licensing systems"""
    license_data = load_license()
    if not license_data:
        return False, "No license found", False

    email = license_data.get("email")
    key = license_data.get("key")

    if not email or not key:
        return False, "Invalid license file format", False

    # First try the new asymmetric validation system
    try:
        # Use direct file import to bypass normal import issues
        import sys
        import importlib.util
        ROOT_LICENSE_PATH = os.path.dirname(os.path.abspath(__file__))
        expected_path = os.path.join(ROOT_LICENSE_PATH, "license_core_new.py")
        spec = importlib.util.spec_from_file_location("license_core_new", expected_path)
        license_core_new_module = importlib.util.module_from_spec(spec)
        sys.modules["license_core_new"] = license_core_new_module
        spec.loader.exec_module(license_core_new_module)
        
        validate_license_key = getattr(license_core_new_module, 'validate_license_key')
        is_valid, message, license_data_new = validate_license_key(email, key)
        if is_valid:
            # New asymmetric system validation passed
            license_type = license_data_new.get("type", "")
            if license_type == "permanent":
                return True, "Asymmetric license valid - permanent", False
            elif license_type == "trial":
                return True, "Asymmetric license valid - trial", False
    except Exception as e:
        print(f"New asymmetric validation failed, trying old system: {e}")
        # Fall back to old system if new system fails
        pass

    # If new system didn't validate, try old system
    return False, "No valid license found", False

def activate_license_file(license_file_path: str) -> tuple[bool, str]:
    """
    Activate license using a .lic file (asymmetric licensing)

    Args:
        license_file_path: Path to the .lic license file

    Returns:
        tuple[bool, str]: (success, message)
    """
    try:
        # Use direct file import to bypass normal import issues
        import sys
        import importlib.util
        import shutil
        import os
        from pathlib import Path
        
        ROOT_LICENSE_PATH = os.path.dirname(os.path.abspath(__file__))
        expected_path = os.path.join(ROOT_LICENSE_PATH, "license_core_new.py")
        spec = importlib.util.spec_from_file_location("license_core_new", expected_path)
        license_core_new_module = importlib.util.module_from_spec(spec)
        sys.modules["license_core_new"] = license_core_new_module
        spec.loader.exec_module(license_core_new_module)
        
        validate_license = getattr(license_core_new_module, 'validate_license')

        # Validate the license file
        is_valid, message, license_data = validate_license(license_file_path)

        if is_valid:
            # Extract email from license data
            email = license_data.get("email", "")
            license_type = license_data.get("type", "")

            if email:
                # Delete any existing trial state to prevent conflicts
                delete_trial_state()

                # Create a permanent copy of the license file in the licensing directory
                # Use the email to create a standardized filename
                email_prefix = email.split('@')[0]
                if license_type == "permanent":
                    destination_filename = f"{email_prefix}_permanent.lic"
                else:
                    destination_filename = f"{email_prefix}_trial.lic"

                destination_path = os.path.join(ROOT_LICENSE_PATH, destination_filename)

                # Copy the license file to the permanent location
                shutil.copy2(license_file_path, destination_path)

                # Save the license using the email as key (for backward compatibility)
                # The actual validation will use the .lic file
                if save_license(email, "asymmetric_license"):
                    return True, f"✅ License file activated successfully ({license_type})"
                else:
                    return False, "❌ Failed to save license information"
            else:
                return False, "❌ Invalid license file: No email found"
        else:
            return False, f"❌ Invalid license file: {message}"

    except Exception as e:
        return False, f"❌ Error activating license file: {str(e)}"

def ensure_license_interactive(prompt_func, app_version: str = "1.0"):
    """
    Interactive license validation with proper error handling
    Returns True if license is valid, False otherwise
    """
    # First check if we have a valid license file
    is_valid, message, integrity_failed = validate_license_production_with_guardrails(app_version)
    if is_valid:
        print(f"✅ License validation passed: {message}")
        if integrity_failed:
            print("⚠️ Installation integrity check failed (non-blocking)")
        return True

    print(f"⚠️ License check failed: {message}")

    # If no valid license, prompt user
    max_attempts = 3
    attempt = 1

    while attempt <= max_attempts:
        print(f"License required (Attempt {attempt}/{max_attempts})")

        result = prompt_func()
        if not result:
            print("License input cancelled")
            return False

        email, key = result

        # Validate the entered license with trial support
        is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
        if is_valid:
            if save_license(email, key):
                print("✅ License saved and validated successfully")
                if integrity_failed:
                    print("⚠️ Installation integrity check failed (non-blocking)")
                return True
            else:
                print("❌ Failed to save license")
        else:
            print(f"❌ {message}")

        attempt += 1

    print("❌ Maximum license attempts exceeded")
    return False

def ensure_license_strict(prompt_func, app_version: str = "1.0"):
    """
    Strict license validation - exits application if no valid license
    """
    is_valid, message, integrity_failed = validate_license_production_with_guardrails(app_version)
    if is_valid:
        print(f"✅ License validation passed: {message}")
        if integrity_failed:
            print("⚠️ Installation integrity check failed (non-blocking)")
        return True

    print(f"❌ License validation failed: {message}")

    # Try to get a license interactively
    if ensure_license_interactive(prompt_func, app_version):
        return True

    # If still no valid license, exit
    print("❌ No valid license found. Application cannot continue.")
    return False

def ensure_license_with_trial_interactive(prompt_func, app_version: str = "1.0"):
    """
    Interactive license validation with trial support
    Returns True if license or trial is valid, False otherwise
    """
    # Check if we have a valid license file
    license_data = load_license()
    if license_data:
        email = license_data.get("email")
        key = license_data.get("key")

        if email and key:
            is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
            if is_valid:
                print(f"✅ {message}")
                return True

    # Check for active trial
    trial_active, trial_message, days_remaining = get_trial_status()
    if trial_active:
        print(f"✅ {trial_message}")
        return True

    # No valid license or trial, prompt user
    print("License or trial required")

    # Offer trial option first
    result = prompt_func(trial_mode=True)
    if result:
        if isinstance(result, tuple) and len(result) == 2:
            # License activation
            email, key = result
            is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
            if is_valid:
                if save_license(email, key):
                    print("✅ License saved and validated successfully")
                    return True
            else:
                print(f"❌ {message}")
        elif isinstance(result, str):
            # Trial activation
            email = result
            trial_started = start_trial(email)
            if trial_started:
                print("✅ Trial started successfully")
                # Verify trial was actually started
                trial_active, trial_message, days_remaining = get_trial_status()
                return True
            else:
                print("❌ Failed to start trial or trial already active")
                # Check if trial already exists
                trial_active, trial_message, days_remaining = get_trial_status()
                if trial_active:
                    print(f"✅ Using existing trial: {trial_message}")
                    return True

    print("❌ No valid license or trial found")
    return False

def ensure_license_interactive_with_guardrails(prompt_func, app_version: str = "1.0"):
    """
    Interactive license validation with guardrails and trial support
    Returns True if license or trial is valid, False otherwise
    """
    # Check if we have a valid license file with guardrails
    is_valid, message, integrity_failed = validate_license_production_with_guardrails(app_version)
    if is_valid:
        print(f"✅ License validation passed: {message}")
        if integrity_failed:
            print("⚠️ Installation integrity check failed (non-blocking)")
        return True

    # Check for active trial status
    trial_active, trial_message, days_remaining = get_trial_status()

    if trial_active:
        # Trial is active -> allow startup
        print(f"✅ {trial_message}")
        return True

    # No active trial and no valid license - allow interactive attempts to activate license or start trial
    print(f"⚠️ No active trial: {trial_message}")
    print(f"⚠️ License check failed: {message}")

    # If no valid license, prompt user with trial support (max attempts)
    max_attempts = 3
    attempt = 1

    while attempt <= max_attempts:
        print(f"License or trial required (Attempt {attempt}/{max_attempts})")

        result = prompt_func()
        if not result:
            print("License input cancelled")
            return False

        # Handle both license and trial results
        if isinstance(result, tuple) and len(result) == 2:
            # License activation
            email, key = result
            is_valid, message, is_permanent, is_trial_active, is_expired = validate_license_with_trial(email, key, app_version)
            if is_valid:
                if save_license(email, key):
                    print("✅ License saved and validated successfully")
                    if integrity_failed:
                        print("⚠️ Installation integrity check failed (non-blocking)")
                    return True
            else:
                print(f"❌ {message}")
        elif isinstance(result, str):
            # Trial activation (email string)
            email = result
            if start_trial(email):
                print("✅ Trial started successfully")
                return True
            else:
                print("❌ Failed to start trial or trial already active")

        attempt += 1

    print("❌ Maximum license attempts exceeded")
    # CRITICAL: block application startup if no valid license/trial after interactive attempts
    return False

# Trial integrity management functions
def initialize_trial_integrity(app_path: str) -> bool:
    """Initialize trial integrity system"""
    try:
        integrity_manager = TrialIntegrityManager()
        return integrity_manager.create_install_marker(app_path)
    except Exception as e:
        print(f"Warning: Could not initialize trial integrity: {e}")
        return False

def check_trial_integrity(app_path: str) -> tuple[bool, str]:
    """Check trial integrity and detect reinstalls"""
    try:
        integrity_manager = TrialIntegrityManager()
        return integrity_manager.should_reset_trial(app_path)
    except Exception as e:
        print(f"Warning: Could not check trial integrity: {e}")
        return False, "Integrity check failed"

def get_trial_integrity_info() -> dict:
    """Get comprehensive trial integrity information"""
    try:
        integrity_manager = TrialIntegrityManager()
        return integrity_manager.get_trial_integrity_info()
    except Exception as e:
        print(f"Warning: Could not get trial integrity info: {e}")
        return {"error": str(e)}

def cleanup_old_trial_markers(days_threshold: int = 90) -> int:
    """Clean up old trial installation markers"""
    try:
        integrity_manager = TrialIntegrityManager()
        return integrity_manager.cleanup_old_markers(days_threshold)
    except Exception as e:
        print(f"Warning: Could not cleanup old markers: {e}")
        return 0

def start_trial_with_integrity_check(email: str, app_path: str) -> tuple[bool, str]:
    """Start trial with integrity check for reinstall detection"""
    # Check if trial should be reset due to reinstall
    should_reset, reason = check_trial_integrity(app_path)

    if should_reset:
        # Delete existing trial state
        delete_trial_state()
        print(f"Trial reset due to: {reason}")

    # Initialize integrity marker
    initialize_trial_integrity(app_path)

    # Start new trial
    if start_trial(email):
        return True, "Trial started successfully with integrity check"
    else:
        return False, "Failed to start trial"

def validate_trial_with_integrity(email: str, key: str, app_path: str) -> tuple[bool, str, bool, bool]:
    """
    Validate trial with integrity check

    Returns:
        tuple[bool, str, bool, bool]: (is_valid, message, is_permanent, is_expired)
    """
    # Check for reinstall first
    should_reset, reason = check_trial_integrity(app_path)

    if should_reset:
        # Delete existing trial state and return no trial found
        delete_trial_state()
        return False, f"Trial reset due to reinstall: {reason}", False, False

    # Proceed with normal trial validation
    return validate_trial_license(email, key)