# trial_integrity_manager.py
import json
import os
import hashlib
import platform
from pathlib import Path
from datetime import datetime
from typing import Optional, Dict, Any, Tuple

class TrialIntegrityManager:
    """Manages trial integrity across reinstalls and system changes"""
    
    def __init__(self):
        self.system = platform.system().lower()
        self.machine_fingerprint = self._get_machine_fingerprint()
        self.install_markers = self._get_install_markers()
        
    def _get_machine_fingerprint(self) -> str:
        """Get stable machine fingerprint"""
        try:
            from .fails_open_integrity import get_stable_machine_fingerprint
            return get_stable_machine_fingerprint()
        except ImportError:
            import sys
            sys.path.append(os.path.dirname(os.path.abspath(__file__)))
            from fails_open_integrity import get_stable_machine_fingerprint
            return get_stable_machine_fingerprint()
    
    def _get_install_markers(self) -> list:
        """Get possible installation marker locations"""
        markers = []
        
        # Primary marker in user home
        home_dir = Path.home()
        markers.append(home_dir / ".aimms_install_marker.json")
        
        # Application data directory
        if self.system == "windows":
            app_data = Path(os.environ.get("APPDATA", ""))
            markers.append(app_data / "AIMMS" / "install_marker.json")
        elif self.system == "darwin":  # macOS
            app_support = Path.home() / "Library" / "Application Support" / "AIMMS"
            markers.append(app_support / "install_marker.json")
        else:  # Linux and others
            config_dir = Path.home() / ".config" / "aimms"
            markers.append(config_dir / "install_marker.json")
        
        # System registry (Windows only)
        if self.system == "windows":
            markers.append("registry")
        
        return markers
    
    def create_install_marker(self, app_path: str) -> bool:
        """Create installation marker for reinstall detection"""
        marker_data = {
            "install_time": datetime.now().isoformat(),
            "machine_fingerprint": self.machine_fingerprint,
            "app_path_hash": hashlib.sha256(app_path.encode()).hexdigest()[:16],
            "system_info": {
                "platform": platform.platform(),
                "machine": platform.machine(),
                "processor": platform.processor(),
                "version": platform.version()
            },
            "marker_id": hashlib.sha256(f"{self.machine_fingerprint}_{datetime.now().isoformat()}".encode()).hexdigest()[:16]
        }
        
        success_count = 0
        
        # Save to multiple locations
        for location in self.install_markers:
            try:
                if location == "registry":
                    self._save_marker_to_registry(marker_data)
                    success_count += 1
                else:
                    # Ensure directory exists
                    location.parent.mkdir(parents=True, exist_ok=True)
                    
                    # Make file hidden on Windows
                    if self.system == "windows":
                        self._make_hidden_file(location)
                    
                    with open(location, 'w') as f:
                        json.dump(marker_data, f, indent=2)
                    
                    success_count += 1
                    
            except Exception as e:
                print(f"Warning: Could not create install marker at {location}: {e}")
                continue
        
        return success_count > 0
    
    def get_install_marker(self) -> Optional[Dict[str, Any]]:
        """Get installation marker from secure locations"""
        # Try locations in order
        for location in self.install_markers:
            try:
                if location == "registry":
                    marker = self._load_marker_from_registry()
                    if marker:
                        return marker
                else:
                    if location.exists():
                        with open(location, 'r') as f:
                            return json.load(f)
            except Exception:
                continue
        
        return None
    
    def is_reinstall_detected(self, current_app_path: str) -> Tuple[bool, str]:
        """Check if this is a reinstall and determine appropriate action"""
        current_marker = self.get_install_marker()
        
        if not current_marker:
            # No previous install marker - create new one
            self.create_install_marker(current_app_path)
            return False, "First installation detected"
        
        # Compare machine fingerprints
        stored_fingerprint = current_marker.get("machine_fingerprint")
        if stored_fingerprint != self.machine_fingerprint:
            return True, "Machine fingerprint changed - potential hardware migration"
        
        # Compare application path
        stored_path_hash = current_marker.get("app_path_hash")
        current_path_hash = hashlib.sha256(current_app_path.encode()).hexdigest()[:16]
        
        if stored_path_hash != current_path_hash:
            return True, "Application path changed - potential reinstall"
        
        # Check if installation is very recent (within 5 minutes)
        # This might indicate a quick reinstall/rollback
        install_time = datetime.fromisoformat(current_marker["install_time"])
        time_diff = datetime.now() - install_time
        
        if time_diff.total_seconds() < 300:  # 5 minutes
            return True, "Very recent installation - potential quick reinstall"
        
        # Installation appears legitimate
        return False, "Legitimate installation detected"
    
    def should_reset_trial(self, current_app_path: str) -> Tuple[bool, str]:
        """Determine if trial should be reset due to reinstall"""
        is_reinstall, reason = self.is_reinstall_detected(current_app_path)
        
        if not is_reinstall:
            return False, "No reinstall detected"
        
        # For certain types of reinstalls, we should be more lenient
        if "hardware migration" in reason.lower():
            return False, "Hardware migration allowed - trial preserved"
        
        if "very recent" in reason.lower():
            return True, "Quick reinstall detected - trial reset"
        
        if "application path changed" in reason.lower():
            return True, "Application path changed - trial reset"
        
        # Default: be conservative and reset trial
        return True, f"Reinstall detected: {reason} - trial reset"
    
    def update_install_marker(self, app_path: str) -> bool:
        """Update installation marker with current information"""
        marker_data = {
            "install_time": datetime.now().isoformat(),
            "machine_fingerprint": self.machine_fingerprint,
            "app_path_hash": hashlib.sha256(app_path.encode()).hexdigest()[:16],
            "system_info": {
                "platform": platform.platform(),
                "machine": platform.machine(),
                "processor": platform.processor(),
                "version": platform.version()
            },
            "marker_id": hashlib.sha256(f"{self.machine_fingerprint}_{datetime.now().isoformat()}".encode()).hexdigest()[:16],
            "update_count": self._get_update_count() + 1
        }
        
        # Update all existing markers
        success_count = 0
        
        for location in self.install_markers:
            try:
                if location == "registry":
                    self._save_marker_to_registry(marker_data)
                    success_count += 1
                else:
                    # Ensure directory exists
                    location.parent.mkdir(parents=True, exist_ok=True)
                    
                    with open(location, 'w') as f:
                        json.dump(marker_data, f, indent=2)
                    
                    success_count += 1
                    
            except Exception:
                continue
        
        return success_count > 0
    
    def _get_update_count(self) -> int:
        """Get current update count from existing marker"""
        marker = self.get_install_marker()
        if marker and "update_count" in marker:
            return marker["update_count"]
        return 0
    
    def _make_hidden_file(self, file_path: Path):
        """Make file hidden on Windows"""
        try:
            import stat
            file_path.chmod(stat.FILE_ATTRIBUTE_HIDDEN)
        except Exception:
            pass
    
    def _save_marker_to_registry(self, marker_data: Dict[str, Any]):
        """Save installation marker to Windows registry"""
        try:
            import winreg
            
            key_path = r"SOFTWARE\AIMMS\InstallMarker"
            
            with winreg.CreateKey(winreg.HKEY_CURRENT_USER, key_path) as key:
                winreg.SetValueEx(key, "MarkerData", 0, winreg.REG_SZ, json.dumps(marker_data))
                winreg.SetValueEx(key, "LastUpdated", 0, winreg.REG_SZ, datetime.now().isoformat())
                
        except Exception as e:
            raise Exception(f"Registry marker save failed: {e}")
    
    def _load_marker_from_registry(self) -> Optional[Dict[str, Any]]:
        """Load installation marker from Windows registry"""
        try:
            import winreg
            
            key_path = r"SOFTWARE\AIMMS\InstallMarker"
            
            with winreg.OpenKey(winreg.HKEY_CURRENT_USER, key_path) as key:
                marker_data_str = winreg.QueryValueEx(key, "MarkerData")[0]
                return json.loads(marker_data_str)
                
        except Exception:
            return None
    
    def get_trial_integrity_info(self) -> Dict[str, Any]:
        """Get comprehensive trial integrity information"""
        marker = self.get_install_marker()
        
        if not marker:
            return {
                "has_install_marker": False,
                "install_time": None,
                "machine_fingerprint": self.machine_fingerprint,
                "app_path_hash": None,
                "is_reinstall": False,
                "reinstall_reason": None,
                "update_count": 0
            }
        
        current_app_path = os.path.dirname(os.path.abspath(__file__))
        is_reinstall, reason = self.is_reinstall_detected(current_app_path)
        
        return {
            "has_install_marker": True,
            "install_time": marker.get("install_time"),
            "machine_fingerprint": self.machine_fingerprint,
            "app_path_hash": marker.get("app_path_hash"),
            "is_reinstall": is_reinstall,
            "reinstall_reason": reason,
            "update_count": marker.get("update_count", 0),
            "stored_machine_fingerprint": marker.get("machine_fingerprint"),
            "stored_app_path_hash": marker.get("app_path_hash")
        }
    
    def cleanup_old_markers(self, days_threshold: int = 90) -> int:
        """Clean up installation markers older than specified days"""
        cleaned_count = 0
        cutoff_date = datetime.now() - datetime.timedelta(days=days_threshold)
        
        for location in self.install_markers:
            try:
                if location == "registry":
                    # Skip registry cleanup for now
                    continue
                else:
                    if location.exists():
                        with open(location, 'r') as f:
                            marker_data = json.load(f)
                        
                        install_time = datetime.fromisoformat(marker_data["install_time"])
                        
                        if install_time < cutoff_date:
                            try:
                                location.unlink()
                                cleaned_count += 1
                            except Exception:
                                pass
                                
            except Exception:
                continue
        
        return cleaned_count