# secure_revocation.py
import hashlib
import json
import os
from pathlib import Path
from datetime import datetime

class SecureRevocationManager:
    """Secure revocation manager that stores license hashes instead of plain keys"""

    def __init__(self):
        self.revocation_file = Path(__file__).parent / "revoked_licenses.json"
        self.revocation_salt = "AIMMS_REVOCATION_SALT_2026"  # Internal salt
        self.load_revocations()

    def load_revocations(self):
        """Load revoked license hashes (secure, no plain keys)"""
        try:
            if self.revocation_file.exists():
                with open(self.revocation_file, 'r') as f:
                    data = json.load(f)
                    self.revoked_hashes = set(data.get("revoked_hashes", []))
                    self.blocked_versions = set(data.get("blocked_versions", []))
            else:
                self.revoked_hashes = set()
                self.blocked_versions = set()
        except Exception:
            # Silent failure - revocation is optional
            self.revoked_hashes = set()
            self.blocked_versions = set()

    def hash_license_key(self, license_key: str) -> str:
        """Hash license key with internal salt for revocation storage"""
        normalized_key = self._normalize_key(license_key)
        payload = f"{normalized_key}|{self.revocation_salt}".encode("utf-8")
        return hashlib.sha256(payload).hexdigest()

    def is_revoked(self, license_key: str, app_version: str) -> bool:
        """Check if license is revoked using secure hash"""
        # Hash the provided license key
        key_hash = self.hash_license_key(license_key)

        # Check version blocking
        if app_version in self.blocked_versions:
            return True

        # Check hash-based revocation
        return key_hash in self.revoked_hashes

    def add_revocation(self, license_key: str, app_version: str):
        """Add license to revocation list using secure hash"""
        key_hash = self.hash_license_key(license_key)
        self.revoked_hashes.add(key_hash)

        # Save to file for future versions (forward-only)
        try:
            data = {
                "revoked_hashes": list(self.revoked_hashes),
                "blocked_versions": list(self.blocked_versions),
                "last_updated": datetime.now().isoformat(),
                "note": "Hashes only - no plain license keys stored"
            }
            with open(self.revocation_file, 'w') as f:
                json.dump(data, f, indent=2)
        except Exception:
            # Silent failure - revocation is optional
            pass

    def _normalize_key(self, key: str) -> str:
        """Normalize license key for consistent hashing"""
        return key.replace("-", "").strip().lower()

    def block_version(self, app_version: str):
        """Block a specific app version"""
        self.blocked_versions.add(app_version)
        try:
            data = {
                "revoked_hashes": list(self.revoked_hashes),
                "blocked_versions": list(self.blocked_versions),
                "last_updated": datetime.now().isoformat(),
                "note": "Hashes only - no plain license keys stored"
            }
            with open(self.revocation_file, 'w') as f:
                json.dump(data, f, indent=2)
        except Exception:
            # Silent failure - revocation is optional
            pass