# license_core.py
import hashlib
import base64
import textwrap
import datetime
from pathlib import Path
import json
import os
import platform
import subprocess

def normalize_key(key: str) -> str:
    return key.replace("-", "").strip()


def validate_trial_license(email: str, key: str) -> tuple[bool, str, bool, bool]:
    """
    Validate license with trial period check and machine binding

    Returns:
        tuple[bool, str, bool, bool]: (is_valid, message, is_permanent, is_expired)
    """
    # Check if permanent license first using new asymmetric validation
    try:
        from license_core_new import validate_license_key
        is_valid, message, license_data = validate_license_key(email, key)
        if is_valid:
            license_type = license_data.get("type", "")
            if license_type == "permanent":
                return True, "Permanent license", True, False
    except ImportError:
        # Fallback to old method if new validation system is not available
        pass

    # Check trial license using old method (keep existing trial system intact)
    trial_data = load_trial_state()
    if not trial_data:
        return False, "No trial or license found", False, False

    # Check machine fingerprint binding
    current_machine = get_stable_machine_fingerprint()
    stored_machine = trial_data.get('machine_fingerprint')

    if stored_machine and current_machine != stored_machine:
        return False, "Trial not bound to this machine", False, False

    # Check trial expiration
    current_date = datetime.datetime.now()
    trial_end_date = datetime.datetime.fromisoformat(trial_data['end_date'])

    if current_date > trial_end_date:
        return False, "Trial expired", False, True

    return True, "Trial active", False, False

# Use consistent relative imports for package structure
from .trial_state_manager import TrialStateManager

# Global trial state manager instance
_trial_state_manager = TrialStateManager()

def load_trial_state():
    """Load trial state using secure trial state manager"""
    return _trial_state_manager.load_trial_state()

def save_trial_state(email: str, days: int = 30):
    """Save trial state using secure trial state manager"""
    trial_data = {
        "email": email,
        "start_date": datetime.datetime.now().isoformat(),
        "end_date": (datetime.datetime.now() + datetime.timedelta(days=days)).isoformat(),
        "machine_fingerprint": get_stable_machine_fingerprint(),
        "install_path_hash": hashlib.sha256(os.path.abspath(__file__).encode()).hexdigest()[:8]
    }

    return _trial_state_manager.save_trial_state(trial_data)

def delete_trial_state():
    """Delete trial state from all secure locations"""
    return _trial_state_manager.delete_trial_state()

def trial_state_exists():
    """Check if trial state exists in any secure location"""
    return _trial_state_manager.trial_state_exists()

# Import machine fingerprinting from fails_open_integrity.py
from .fails_open_integrity import get_stable_machine_fingerprint